<?php
class Users extends Controller
{
    private $userModel;

    public function __construct()
    {
        $this->userModel = $this->model('User');
    }

    public function login()
    {
        // Check for POST
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            // Process form
            // Sanitize POST data
            $_POST = filter_input_array(INPUT_POST, FILTER_SANITIZE_STRING);

            // Init data
            $data = [
                'usuario' => trim($_POST['usuario']),
                'password' => trim($_POST['password']),
                'usuario_err' => '',
                'password_err' => '',
            ];

            // Validate Email
            if (empty($data['usuario'])) {
                $data['usuario_err'] = 'Por favor ingrese usuario';
            }

            // Validate Password
            if (empty($data['password'])) {
                $data['password_err'] = 'Por favor ingrese contraseña';
            }

            // Check for user/email
            if ($this->userModel->findUserByUsername($data['usuario'])) {
                // User found
            } else {
                // User not found
                $data['usuario_err'] = 'No se encontró el usuario';
            }

            // Make sure errors are empty
            if (empty($data['usuario_err']) && empty($data['password_err'])) {
                // Validated
                // Check and set logged in user
                $loggedInUser = $this->userModel->login($data['usuario'], $data['password']);

                if ($loggedInUser) {
                    // Create Session
                    $this->createUserSession($loggedInUser);
                } else {
                    $data['password_err'] = 'Contraseña incorrecta';
                    $this->view('users/login', $data);
                }
            } else {
                // Load view with errors
                $this->view('users/login', $data);
            }

        } else {
            // Init data
            $data = [
                'usuario' => '',
                'password' => '',
                'usuario_err' => '',
                'password_err' => '',
            ];

            // Load view
            $this->view('users/login', $data);
        }
    }

    public function createUserSession($user)
    {
        $_SESSION['user_id'] = $user->id;
        $_SESSION['user_email'] = $user->usuario;
        $_SESSION['user_name'] = $user->nombre;
        redirect('pages/index');
    }

    public function logout()
    {
        unset($_SESSION['user_id']);
        unset($_SESSION['user_email']);
        unset($_SESSION['user_name']);
        session_destroy();
        redirect('users/login');
    }

    public function profile()
    {
        if (!isLoggedIn()) {
            redirect('users/login');
        }

        // Get current user data
        $user = $this->userModel->getUserById($_SESSION['user_id']);

        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            // Sanitize POST data
            $_POST = filter_input_array(INPUT_POST, FILTER_SANITIZE_STRING);

            $data = [
                'user' => $user,
                'nombre' => trim($_POST['nombre']),
                'usuario' => trim($_POST['usuario']),
                'current_password' => trim($_POST['current_password']),
                'new_password' => trim($_POST['new_password']),
                'confirm_password' => trim($_POST['confirm_password']),
                'nombre_err' => '',
                'usuario_err' => '',
                'current_password_err' => '',
                'new_password_err' => '',
                'confirm_password_err' => '',
                'success_msg' => ''
            ];

            // Validate nombre
            if (empty($data['nombre'])) {
                $data['nombre_err'] = 'Por favor ingrese su nombre';
            }

            // Validate usuario
            if (empty($data['usuario'])) {
                $data['usuario_err'] = 'Por favor ingrese un nombre de usuario';
            } else {
                // Check if username is taken (excluding current user)
                if ($this->userModel->checkUsernameExists($data['usuario'], $_SESSION['user_id'])) {
                    $data['usuario_err'] = 'Este nombre de usuario ya está en uso';
                }
            }

            // If password fields are filled, validate password change
            if (!empty($data['current_password']) || !empty($data['new_password']) || !empty($data['confirm_password'])) {
                // Validate current password
                if (empty($data['current_password'])) {
                    $data['current_password_err'] = 'Por favor ingrese su contraseña actual';
                } else {
                    // Verify current password
                    if (!password_verify($data['current_password'], $user->password)) {
                        $data['current_password_err'] = 'Contraseña actual incorrecta';
                    }
                }

                // Validate new password
                if (empty($data['new_password'])) {
                    $data['new_password_err'] = 'Por favor ingrese una nueva contraseña';
                } elseif (strlen($data['new_password']) < 6) {
                    $data['new_password_err'] = 'La contraseña debe tener al menos 6 caracteres';
                }

                // Validate confirm password
                if (empty($data['confirm_password'])) {
                    $data['confirm_password_err'] = 'Por favor confirme la nueva contraseña';
                } else {
                    if ($data['new_password'] != $data['confirm_password']) {
                        $data['confirm_password_err'] = 'Las contraseñas no coinciden';
                    }
                }
            }

            // Make sure errors are empty
            if (
                empty($data['nombre_err']) && empty($data['usuario_err']) &&
                empty($data['current_password_err']) && empty($data['new_password_err']) &&
                empty($data['confirm_password_err'])
            ) {

                // Update user information
                $updateData = [
                    'id' => $_SESSION['user_id'],
                    'nombre' => $data['nombre'],
                    'usuario' => $data['usuario']
                ];

                if ($this->userModel->updateUser($updateData)) {
                    // Update session
                    $_SESSION['user_name'] = $data['nombre'];
                    $_SESSION['user_email'] = $data['usuario'];

                    // Update password if provided
                    if (!empty($data['new_password'])) {
                        $this->userModel->updatePassword($_SESSION['user_id'], $data['new_password']);
                    }

                    $data['success_msg'] = 'Perfil actualizado exitosamente';
                    // Refresh user data
                    $data['user'] = $this->userModel->getUserById($_SESSION['user_id']);
                } else {
                    $data['nombre_err'] = 'Algo salió mal, por favor intente de nuevo';
                }
            }

            $this->view('users/profile', $data);
        } else {
            $data = [
                'user' => $user,
                'nombre' => $user->nombre,
                'usuario' => $user->usuario,
                'current_password' => '',
                'new_password' => '',
                'confirm_password' => '',
                'nombre_err' => '',
                'usuario_err' => '',
                'current_password_err' => '',
                'new_password_err' => '',
                'confirm_password_err' => '',
                'success_msg' => ''
            ];

            $this->view('users/profile', $data);
        }
    }
}

