<?php
class Sale
{
    private $db;

    public function __construct()
    {
        $this->db = new Database;
    }

    public function createSale($data, $cart)
    {
        // Start Transaction
        $this->db->beginTransaction();

        try {
            // 1. Insert into ventas
            $this->db->query('INSERT INTO ventas (cliente_id, usuario_id, total, tipo_pago) VALUES(:cliente_id, :usuario_id, :total, :tipo_pago)');
            $this->db->bind(':cliente_id', $data['cliente_id']);
            $this->db->bind(':usuario_id', $data['usuario_id']);
            $this->db->bind(':total', $data['total']);
            $this->db->bind(':tipo_pago', $data['tipo_pago']);

            if (!$this->db->execute()) {
                throw new Exception("Error creating sale header");
            }

            $venta_id = $this->db->lastInsertId();

            // 2. Insert details and update stock
            foreach ($cart as $item) {
                $this->db->query('INSERT INTO detalle_ventas (venta_id, producto_id, cantidad, precio_unitario, subtotal) VALUES(:venta_id, :producto_id, :cantidad, :precio_unitario, :subtotal)');
                $this->db->bind(':venta_id', $venta_id);
                $this->db->bind(':producto_id', $item['id']);
                $this->db->bind(':cantidad', $item['cantidad']);
                $this->db->bind(':precio_unitario', $item['precio']);
                $this->db->bind(':subtotal', $item['subtotal']);

                if (!$this->db->execute()) {
                    throw new Exception("Error inserting sale detail");
                }

                // Update Stock
                $this->db->query('UPDATE productos SET stock = stock - :cantidad WHERE id = :id');
                $this->db->bind(':cantidad', $item['cantidad']);
                $this->db->bind(':id', $item['id']);

                if (!$this->db->execute()) {
                    throw new Exception("Error updating stock");
                }
            }

            // Commit
            $this->db->commit();
            return $venta_id;

        } catch (Exception $e) {
            $this->db->rollBack();
            return false;
        }
    }

    public function getSaleById($id)
    {
        $this->db->query('SELECT v.*, c.nombre as cliente_nombre, c.ruc_cedula, u.nombre as vendedor_nombre FROM ventas v 
                          LEFT JOIN clientes c ON v.cliente_id = c.id
                          LEFT JOIN usuarios u ON v.usuario_id = u.id
                          WHERE v.id = :id');
        $this->db->bind(':id', $id);
        return $this->db->single();
    }

    public function getSaleDetails($id)
    {
        $this->db->query('SELECT dv.*, p.nombre as producto_nombre FROM detalle_ventas dv
                          JOIN productos p ON dv.producto_id = p.id
                          WHERE dv.venta_id = :id');
        $this->db->bind(':id', $id);
        return $this->db->resultSet();
    }

    public function countSalesToday()
    {
        $this->db->query('SELECT COUNT(*) as count FROM ventas WHERE DATE(fecha) = CURDATE()');
        $row = $this->db->single();
        return $row->count;
    }

    public function sumSalesToday()
    {
        $this->db->query('SELECT SUM(total) as total FROM ventas WHERE DATE(fecha) = CURDATE()');
        $row = $this->db->single();
        return $row->total;
    }

    public function getRecentSales($limit = 5)
    {
        $this->db->query('SELECT v.*, c.nombre as cliente FROM ventas v 
                          LEFT JOIN clientes c ON v.cliente_id = c.id
                          ORDER BY v.fecha DESC LIMIT :limit');
        $this->db->bind(':limit', $limit);
        return $this->db->resultSet();
    }

    // Métodos para reportes
    public function getDailySales()
    {
        $this->db->query('SELECT v.*, c.nombre as cliente_nombre, c.ruc_cedula, u.nombre as vendedor_nombre 
                          FROM ventas v 
                          LEFT JOIN clientes c ON v.cliente_id = c.id
                          LEFT JOIN usuarios u ON v.usuario_id = u.id
                          WHERE DATE(v.fecha) = CURDATE()
                          ORDER BY v.fecha DESC');
        return $this->db->resultSet();
    }

    public function getMonthlySales()
    {
        $this->db->query('SELECT DATE(v.fecha) as fecha, COUNT(*) as num_ventas, SUM(v.total) as total_dia
                          FROM ventas v
                          WHERE YEAR(v.fecha) = YEAR(CURDATE()) AND MONTH(v.fecha) = MONTH(CURDATE())
                          GROUP BY DATE(v.fecha)
                          ORDER BY fecha DESC');
        return $this->db->resultSet();
    }

    public function getAnnualSales()
    {
        $this->db->query('SELECT MONTH(v.fecha) as mes, COUNT(*) as num_ventas, SUM(v.total) as total_mes
                          FROM ventas v
                          WHERE YEAR(v.fecha) = YEAR(CURDATE())
                          GROUP BY MONTH(v.fecha)
                          ORDER BY mes ASC');
        return $this->db->resultSet();
    }

    public function countSalesThisMonth()
    {
        $this->db->query('SELECT COUNT(*) as count FROM ventas 
                          WHERE YEAR(fecha) = YEAR(CURDATE()) AND MONTH(fecha) = MONTH(CURDATE())');
        $row = $this->db->single();
        return $row->count;
    }

    public function sumSalesThisMonth()
    {
        $this->db->query('SELECT SUM(total) as total FROM ventas 
                          WHERE YEAR(fecha) = YEAR(CURDATE()) AND MONTH(fecha) = MONTH(CURDATE())');
        $row = $this->db->single();
        return $row->total;
    }

    public function countSalesThisYear()
    {
        $this->db->query('SELECT COUNT(*) as count FROM ventas WHERE YEAR(fecha) = YEAR(CURDATE())');
        $row = $this->db->single();
        return $row->count;
    }

    public function sumSalesThisYear()
    {
        $this->db->query('SELECT SUM(total) as total FROM ventas WHERE YEAR(fecha) = YEAR(CURDATE())');
        $row = $this->db->single();
        return $row->total;
    }
}

